/***************************************************************************//**
 *   @file   ADXL345.c
 *   @brief  Implementation of ADXL345 Driver.
 *   @author DBogdan (dragos.bogdan@analog.com)
********************************************************************************
 * Copyright 2012(c) Analog Devices, Inc.
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *  - Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *  - Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *  - Neither the name of Analog Devices, Inc. nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *  - The use of this software may or may not infringe the patent rights
 *    of one or more patent holders.  This license does not release you
 *    from the requirement that you obtain separate licenses from these
 *    patent holders to use this software.
 *  - Use of the software either in source or binary form, must be run
 *    on or directly connected to an Analog Devices Inc. component.
 *
 * THIS SOFTWARE IS PROVIDED BY ANALOG DEVICES "AS IS" AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, NON-INFRINGEMENT,
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL ANALOG DEVICES BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, INTELLECTUAL PROPERTY RIGHTS, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
********************************************************************************
 *   SVN Revision: $WCREV$
*******************************************************************************/

/******************************************************************************/
/***************************** Include Files **********************************/
/******************************************************************************/
#include "ADXL345.h"		// ADXL345 definitions.
#include "driver/spi_lite.h"	// Communication definitions.

/******************************************************************************/
/************************ Variables Definitions *******************************/
/******************************************************************************/
char selectedRange     = 0;
char fullResolutionSet = 0;

/******************************************************************************/
/************************ Functions Definitions *******************************/
/******************************************************************************/

/***************************************************************************//**
 * @brief Initializes the communication peripheral and checks if the ADXL345
 *		  part is present.
 *
 *
 * @return status      - Result of the initialization procedure.
 *                       Example: -1 - I2C/SPI peripheral was not initialized or
 *                                     ADXL345 part is not present.
 *                                 0 - I2C/SPI peripheral is initialized and 
 *                                     ADXL345 part is present.
*******************************************************************************/
/*! Reads the value of a register. */
uint8 ADXL345_GetRegisterValue(uint8 registerAddress)
{
	uint8 GetReg[1] = {0};
	spiADXL345(registerAddress, READ, 1, GetReg);
	return GetReg[0];
}

/*! Writes data into a register. */
void ADXL345_SetRegisterValue(uint8 registerAddress, uint8 registerValue)
{
	uint8 GetReg[1];

	GetReg[0]= registerValue;
	spiADXL345(registerAddress, WRITE, 1, GetReg);

}

void ADXL345_Init()
{
    uint8 status = 1;

    spi_init();       // SPI clock edge.

    selectedRange = 2; // Measurement Range: +/- 2g (reset default).
    fullResolutionSet = 0;

    //Set to measurement mode
    ADXL345_SetPowerMode(1);
    //set measurement rate to 200Hz
    ADXL345_SetRegisterValue(ADXL345_BW_RATE, 0b1011);

}

/***************************************************************************//**
 * @brief Places the device into standby/measure mode.
 *
 * @param pwrMode - Power mode.
 *			Example: 0x0 - standby mode.
 *				 0x1 - measure mode.
 *
 * @return None.
*******************************************************************************/
void ADXL345_SetPowerMode(uint8 pwrMode)
{
    uint8 oldPowerCtl = 0;
    uint8 newPowerCtl =0;
    
    oldPowerCtl = ADXL345_GetRegisterValue(ADXL345_POWER_CTL);
    //spiADXL345(ADXL345_POWER_CTL, READ, 1, oldPowerCtl);
    newPowerCtl = oldPowerCtl & ~ADXL345_PCTL_MEASURE;
    newPowerCtl = newPowerCtl | (pwrMode * ADXL345_PCTL_MEASURE);
    ADXL345_SetRegisterValue(ADXL345_POWER_CTL, newPowerCtl);
    //spiADXL345(ADXL345_POWER_CTL, WRITE, 1, newPowerCtl);
}

/***************************************************************************//**
 * @brief Reads the raw output data of each axis.
 *
 * @param x - X-axis's output data.
 * @param y - Y-axis's output data.
 * @param z - Z-axis's output data.
 *
 * @return None.
*******************************************************************************/
void ADXL345_GetXyz(sint16* x,
		    sint16* y,
		    sint16* z)
{
    uint8 readBuffer[7]   = {0, 0, 0, 0, 0, 0, 0};
    
    spiADXL345(ADXL345_DATAX0, READ, 6, readBuffer);

        *x = ((sint16)readBuffer[1] << 8) | readBuffer[0];
        *y = ((sint16)readBuffer[3] << 8) | readBuffer[2];
        *z = ((sint16)readBuffer[5] << 8) | readBuffer[4];
}
