/******************************************************************************
 * Author: 2015 Malcolm Yeoman
 *
 * FileName: spi_lite.c
 *
 * Description: writes a byte out onto the SPI OUT port, and reads a byte from
 *              the SPI IN port
 *
 *******************************************************************************/
#include "driver/spi_lite.h"		// ADXL345 definitions.

void ICACHE_FLASH_ATTR spi_init(void)
{
    //Disable interrupts
    ETS_GPIO_INTR_DISABLE();

    //Set pin functions
    PIN_FUNC_SELECT(PERIPHS_IO_MUX_MTDI_U, FUNC_GPIO12);
    PIN_FUNC_SELECT(PERIPHS_IO_MUX_MTCK_U, FUNC_GPIO13);
    PIN_FUNC_SELECT(PERIPHS_IO_MUX_MTMS_U, FUNC_GPIO14);
    PIN_FUNC_SELECT(PERIPHS_IO_MUX_GPIO4_U, FUNC_GPIO4);

    //enable pull ups
    PIN_PULLUP_EN(PERIPHS_IO_MUX_MTDI_U);
    PIN_PULLUP_EN(PERIPHS_IO_MUX_MTCK_U);
    PIN_PULLUP_EN(PERIPHS_IO_MUX_MTMS_U);
    PIN_PULLUP_EN(PERIPHS_IO_MUX_GPIO4_U);

    //Turn interrupt back on
    ETS_GPIO_INTR_ENABLE();
}


void ICACHE_FLASH_ATTR spiPutByte(uint8 c) {
    uint8 ret;
    uint8 mask;

    mask = 0x80;                						//Initialize to write and read bit 7
    ret = 0;                    						//Initialize read byte with 0

    do  {
        if (c & mask)                			  		//Clock out current bit onto SPI Out line
        	{
        	GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_OUT_N), 1);
        	}
        else
        	{
        	GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_OUT_N), 0);
        	}
        GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CLK_N), 0);                //Start clock bit
        os_delay_us(1);                      				  //Ensure minimum delay of 500nS between SPI Clock Low and SPI Clock High
        GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CLK_N), 1);                //End clock bit
        mask = mask >> 1;           				  //Shift mask so that next bit is written and read from SPI lines
        os_delay_us(1);                      				  //Ensure minimum delay of 1000ns between bits
    } while (mask != 0);

}

uint8 ICACHE_FLASH_ATTR spiGetByte() {
    uint8 ret;
    uint8 mask;

    mask = 0x80;                						//Initialize to write and read bit 7
    ret = 0;                    						//Initialize read byte with 0

    do  {
        GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CLK_N), 0);                //Start clock bit
        os_delay_us(1);                      				  //Ensure minimum delay of 500nS between SPI Clock Low and SPI Clock High
        GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CLK_N), 1);                //End clock bit
        if (GPIO_INPUT_GET(SPI_IN_N)) ret |= mask;    //Read current bit fromSPI In line
        mask = mask >> 1;           				  //Shift mask so that next bit is written and read from SPI lines
        os_delay_us(1);                      				  //Ensure minimum delay of 1000ns between bits
    } while (mask != 0);


    return ret;
}

void ICACHE_FLASH_ATTR spiPutAddr(uint8 addr, uint8 readwrite, uint8 datalen) {
	//readwrite=READ or WRITE
	//datalen = number of data bytes

    uint8 ret;
    uint8 mask;

    //include r/w bit
    addr |= (readwrite << 7);

    //set multibyte bit
    if (datalen > 1) addr |= (readwrite << 6);

    mask = 0x80;                						//Initialize to write and read bit 7

    do  {
        if (addr & mask)                			  //Clock out current bit onto SPI Out line
        	{
        	GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_OUT_N), 1);
        	}
        else
        	{
        	GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_OUT_N), 0);
        	}
        GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CLK_N), 0);                //Start clock bit
        os_delay_us(1);                      				  //Ensure minimum delay of 500nS between SPI Clock Low and SPI Clock High
        GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CLK_N), 1);                //End clock bit
        mask = mask >> 1;           				  //Shift mask so that next bit is written and read from SPI lines
        os_delay_us(1);                      				  //Ensure minimum delay of 1000ns between bits
    } while (mask != 0);


}

void ICACHE_FLASH_ATTR spiADXL345(uint8 addr, uint8 readwrite, uint8 datalen, uint8 *data) {
	//readwrite=1 for read, readwrite=0 for write
	//datalen = number of data bytes

    uint8 ret;
    uint8 mask;
    uint8 n;
    
    //SPI Mode 3. CS active low. Clock idle 1. data captured on clock's rising edge and data  propagated on falling edge.
    GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CLK_N), 1);

    //Enable SPI communication. The SPI Enable signal must be pulsed low for each message
    GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CS_N), 0);
    
    //Ensure a minimum delay of 500ns between falling edge of SPI Enable signal
    //and falling edge of SPI Clock
    os_delay_us(1);

    spiPutAddr(addr, readwrite, datalen);
    
    //if reading
    if (readwrite == READ)
    {
    	for (n=0;n<datalen;n++)
    	{
    		data[n] = spiGetByte();
    	}
    }
    //if writing
    else
    {
    	spiPutByte(data[0]);
    }

    //Ensure a minimum delay of 750ns between falling edge of SPI Clock signal
    //and rising edge of SPI Enable!
    os_delay_us(1);

    //Disable SPI communication by returning SPI Enable signal to high
    GPIO_OUTPUT_SET(GPIO_ID_PIN(SPI_CS_N), 1);
    
}

